#ifndef _NetworkProxy_NetworkProxy_h
#define _NetworkProxy_NetworkProxy_h

#include <Core/Core.h>

#ifdef PLATFORM_POSIX
#include <arpa/inet.h>
#endif

NAMESPACE_UPP

class NetworkProxy
{
    static const char*  GetErrorMessage(int code); 

 protected:
    enum Phase { 
        START, DNS, REQUEST, REPLY, STARTSSL, 
        SSLHANDSHAKE, FINISHED, SUCCESS, FAILED, 
        EXTENSION 
    };
    enum Command { CONNECT = 1, ACCEPT };
    
    TcpSocket*          socket;
    IpAddrInfo          ipaddrinfo;
    int                 phase;
    int                 command;
    String              packet;
    int                 packet_length;
    int                 proxy_type;
    String              proxy_host;
    int                 proxy_port;
    String              target_host;
    int                 target_port;
    int                 timeout;
    int                 start_time;
    bool                ssl;
    String              error_desc;
    int                 error;

    
    bool                SetError(int code);
    inline void         SetPhase(int p)                             { phase = p; }
    void                StartDns(const String& host, int port);
    void                Dns();  
    bool                Put();
    bool                Get();
    virtual bool        IsEof() = 0;
    virtual void        Init();
    virtual void        Start() = 0;
    virtual void        DoConnect() = 0;
    virtual void        MakeRequest() = 0;
    virtual void        ParseReply() = 0;
    void                CheckConnection();
    Callback            NextStep;
    
public:
    typedef NetworkProxy CLASSNAME;
    
    enum Type { 
        HTTP   = 0x03,
        SOCKS4 = 0x04, 
        SOCKS5 = 0x05,
    };

    enum NetworkProxyErrors {   
        NO_SOCKET_ATTACHED = 10000, 
        HOST_NOT_SPECIFIED, 
        TARGET_NOT_SPECIFIED, 
        DNS_FAILED, 
        CONNECTION_FAILED,
        SSL_FAILED, 
        INVALID_PACKET, 
        SOCKET_FAILURE, ABORTED, 
        CONNECTION_TIMED_OUT
    };
    
    NetworkProxy&       Attach(TcpSocket& sock)                             { socket = &sock; return *this; }
    NetworkProxy&       Host(const String& host)                            { proxy_host = host; return *this; }
    NetworkProxy&       Port(int port)                                      { proxy_port = port; return *this; }
    NetworkProxy&       Timeout(int ms)                                     { timeout = ms; return *this; }
    NetworkProxy&       SSL(bool b = true)                                  { ssl = b; return *this; }

    TcpSocket&          GetSocket() const                                   { return *socket; }
    int                 GetType() const                                     { return proxy_type; }

    bool                Connect(const String& host, int port);
    void                StartConnect(const String& host, int port);
    virtual bool        Do();
    void                Abort()                                             { SetPhase(FAILED); }
    void                Restart()                                           { SetPhase(START); }                                    

    bool                InProgress() const                                  { return phase != FAILED && phase != SUCCESS; }
    bool                IsSuccess() const                                   { return phase == SUCCESS;  }
    bool                IsFailure() const                                   { return phase == FAILED; }   

    int                 GetError() const                                    { return error; }
    String              GetErrorDesc() const                                { return error_desc; }

    static void         Trace(bool verbose = false);    
    NetworkProxy();
    virtual ~NetworkProxy();    
};

class HttpProxy : public NetworkProxy
{
    String              proxy_user;
    String              proxy_password;
    
    bool                IsEof();
    void                Start();
    void                DoConnect();
    void                MakeRequest();
    void                ParseReply();    
    
public:
    typedef HttpProxy CLASSNAME;
    
    enum HttpErrors     { HTTP_CONNECT_FAILED = 10010 };
    
    HttpProxy&          Auth(const String& user, const String& pass)        { proxy_user = user; proxy_password = pass; return *this; }
    bool                Do()                                                { return NetworkProxy::Do(); }
   

    HttpProxy()         { proxy_type = HTTP; }
    HttpProxy(TcpSocket& sock);
};

class SocksProxy : public NetworkProxy
{
    enum Phase          { BOUND = EXTENSION };
    enum Packet         { SOCKS5_HELLO, SOCKS5_AUTH, SOCKS_REQUEST };
    
    int                 packet_type;
    String              proxy_user;
    String              proxy_password;
    bool                bound;
    bool                dns_lookup;
    sockaddr_storage    bound_addr;
    

    bool                IsEof();
    void                Start();
    void                DoConnect();
    void                Socks4Request();
    void                Socks5Request();
    void                MakeRequest()                                       { proxy_type == SOCKS4 ? Socks4Request() : Socks5Request(); }
    void                ParseReply();
    void                ParseBoundAddr();   
public:

    typedef SocksProxy CLASSNAME;
    
    enum SocksErrors {  
        SOCKS4_REQUEST_FAILED = 91, 
        SOCKS4_CLIENT_NOT_REACHABLE,    
        SOCKS4_AUTHENTICATION_FAILED,   
        SOCKS4_ADDRESS_TYPE_NOT_SUPPORTED, 
        SOCKS5_GENERAL_FAILURE = 1, 
        SOCKS5_CONNECTION_NOT_ALLOWED, 
        SOCKS5_NETWORK_UNREACHABLE,
        SOCKS5_TARGET_UNREACHABLE,  
        SOCKS5_CONNECTION_REFUSED, 
        SOCKS5_TTL_EXPIRED,
        SOCKS5_COMMAND_NOT_SUPPORTED, 
        SOCKS5_ADDRESS_TYPE_NOT_SUPPORTED,
        SOCKS5_INVALID_AUTHENTICATION_METHOD = 255, 
        SOCKS5_AUTHENTICATION_FAILED = 256
    };   
    
    SocksProxy&         Auth(const String& user)                            { proxy_user = user; return *this; }
    SocksProxy&         Auth(const String& user, const String& pass)        { proxy_password = pass; return *this; }
    SocksProxy&         Socks4()                                            { proxy_type = SOCKS4; return *this; }
    SocksProxy&         Socks5()                                            { proxy_type = SOCKS5; return *this; }
    SocksProxy&         DnsLookup(bool b = true)                            { dns_lookup = b; return *this; }                                                   
    
    bool                Accept(const String& host, int port);
    void                StartAccept(const String& host, int port);
    bool                Do();
    
    Tuple2<String, int> GetBoundAddr();
    bool                IsBound() const                                     { return phase == BOUND; }
    Callback1<SocksProxy&> WhenBound;
                                    
    SocksProxy();
    SocksProxy(TcpSocket& sock);
};

int ProxyConnect(int type, TcpSocket& socket, const String& proxy_host, int proxy_port,
                const String& target_host, int target_port, const String& user = Null, const String& password = Null,  
                int timeout = 120000, bool ssl = false);
int ProxyAccept(int type, TcpSocket& socket, const String& proxy_host, int proxy_port,
                const String& target_host, int target_port, Callback1<SocksProxy&> whenbound,
                const String& user = Null, const String& password = Null, int timeout = 120000);

END_UPP_NAMESPACE                
#endif
